/**
 * Init - Ponto de Entrada Público
 * 
 * Missão: Gerar trc_id → Enviar evento → Redirecionar para miniapp
 * 
 * Otimizado: Sem operações desnecessárias, código limpo e performático
 */
(function () {
  'use strict';

  // Constantes
  const DEFAULTS = {
    BOT: 'trcsub_bot',
    MODE: 'main',
    PATH: 'app',
    DELAY_MIN: 150,
    DELAY_MAX: 300,
    MAX_WAIT: 5000
  };

  // Debug overlay (apenas quando necessário)
  (function initDebug() {
    try {
      const search = (location.search || '').replace(/amp%3B/gi, '').replace(/amp;/gi, '');
      const params = new URLSearchParams(search);
      const isDebug = params.get('debug') === '1' || localStorage.getItem('trc:init_debug') === '1';
      const isVerbose = params.get('verbose') === '1';
      
      if (!isDebug) return;

      const panel = document.getElementById('trcDbgPanel');
      const out = document.getElementById('trcDbgOut');
      if (!panel || !out) return;

      let seq = 0;
      const buf = [];
      
      function append(text) {
        try {
          const ts = new Date().toISOString();
          out.textContent += `\n[${ts}] ${text}`;
          out.scrollTop = out.scrollHeight;
        } catch {}
      }

      window.__trc_log = function(entry) {
        try {
          const text = typeof entry === 'string' ? entry : JSON.stringify(entry);
          if (panel.style.display === 'block') {
            append(text);
          } else {
            buf.push(text);
          }
        } catch {}
      };

      panel.style.display = 'block';

      // Controles de debug
      try {
        const bar = document.createElement('div');
        bar.style.cssText = 'position:sticky;top:0;display:flex;gap:8px;margin-bottom:8px;background:rgba(0,0,0,0.9);padding:4px';
        
        ['Copiar', 'Baixar', 'Limpar'].forEach((label, i) => {
          const btn = document.createElement('button');
          btn.textContent = label;
          btn.onclick = i === 0 ? () => {
            try { navigator.clipboard.writeText(out.textContent || ''); } catch {}
          } : i === 1 ? () => {
            try {
              const blob = new Blob([out.textContent || ''], { type: 'text/plain' });
              const url = URL.createObjectURL(blob);
              const a = document.createElement('a');
              a.href = url;
              a.download = 'init-logs.txt';
              a.click();
              setTimeout(() => URL.revokeObjectURL(url), 1000);
            } catch {}
          } : () => { out.textContent = ''; };
          bar.appendChild(btn);
        });
        
        panel.insertBefore(bar, out);
      } catch {}

      // Processar buffer
      buf.forEach(append);
      buf.length = 0;
      append(`debug ativado (${search || 'no_qs'})`);
      if (isVerbose) append('verbose=1');

      // Error handlers
      window.addEventListener('error', (ev) => {
        try {
          window.__trc_log({ kind: 'error', message: ev.message, file: ev.filename, line: ev.lineno });
        } catch {}
      });
      window.addEventListener('unhandledrejection', (ev) => {
        try {
          window.__trc_log({ kind: 'unhandled', reason: String(ev.reason) });
        } catch {}
      });
    } catch {}
  })();

  // Logger estruturado (apenas em debug)
  function log(kind, data) {
    if (!window.__trc_log) return;
    try {
      const payload = { kind, seq: (window.__trc_seq = (window.__trc_seq || 0) + 1), t: Math.round(performance.now()) };
      if (data) Object.assign(payload, data);
      window.__trc_log(payload);
    } catch {}
  }

  // Gera trc_id simples
  function generateTrcId() {
    try {
      const ts = Math.floor(Date.now() / 1000);
      const rand = Math.random().toString(36).substring(2, 11);
      return `${ts}_${rand}`;
    } catch {
      return `${Date.now()}_${Math.random().toString(36).substring(2, 9)}`;
    }
  }

  // Obtém trc_id (busca do storage ou gera novo)
  function getTrcId() {
    try {
      // Tentar buscar do storage (pode ter sido salvo pelo hybrid)
      const stored = localStorage.getItem('trc:trc_id') || localStorage.getItem('trc_id');
      if (stored && stored !== 'unknown' && stored.length > 0) {
        log('trc_id_found', { trc_id: stored });
        return stored;
      }
    } catch {}
    
    // Gerar novo
    const newId = generateTrcId();
    log('trc_id_generated', { trc_id: newId });
    return newId;
  }

  // Parse de URL params (single source of truth)
  function parseParams() {
    const search = (location.search || '').replace(/amp%3B/gi, '').replace(/amp;/gi, '');
    const clean = search.startsWith('?') ? search.substring(1) : search;
    const params = new URLSearchParams(clean);
    
    return {
      params,
      debug: params.get('debug') === '1',
      hold: params.has('hold'),
      delay: parseInt(params.get('delay') || 'NaN', 10),
      redirect: params.get('redirect')
    };
  }

  // Carrega configuração
  async function loadConfig() {
    try {
      if (window.__trc_config_ready && typeof window.__trc_config_ready.then === 'function') {
        await window.__trc_config_ready;
      }
      
      const cfg = window.__getTrcConfig?.() || {};
      const tg = cfg.telegram || {};
      const rd = cfg.redirect || {};
      
      return {
        bot: tg.bot || DEFAULTS.BOT,
        mode: tg.mode || DEFAULTS.MODE,
        path: tg.miniapp_path || DEFAULTS.PATH,
        delayMin: rd.delay_ms_min || DEFAULTS.DELAY_MIN,
        delayMax: rd.delay_ms_max || DEFAULTS.DELAY_MAX,
        redirectEnabled: rd.enabled !== false,
        allowOverride: rd.allow_query_override !== false
      };
    } catch {
      return {
        bot: DEFAULTS.BOT,
        mode: DEFAULTS.MODE,
        path: DEFAULTS.PATH,
        delayMin: DEFAULTS.DELAY_MIN,
        delayMax: DEFAULTS.DELAY_MAX,
        redirectEnabled: true,
        allowOverride: true
      };
    }
  }

  // Constrói link do miniapp
  function buildLink(bot, mode, path, trcId) {
    if (!bot || !trcId) return null;
    const encoded = encodeURIComponent(trcId);
    return mode.toLowerCase() === 'direct'
      ? `https://t.me/${bot}/${path}?startapp=${encoded}`
      : `https://t.me/${bot}?startapp=${encoded}`;
  }

  // Coleta query params para o payload
  function collectQueryParams(params) {
    const query = {};
    try {
      for (const [key, value] of params.entries()) {
        if (key && value && !key.startsWith('?')) {
          query[key] = value;
        }
      }
      query.event_time = Math.floor(Date.now() / 1000);
    } catch {}
    return query;
  }

  // Função principal
  async function main() {
    try {
      // 1. Carregar config
      const config = await loadConfig();
      log('config_loaded', { bot: config.bot, mode: config.mode });

      // 2. Parse params
      const { params, debug, hold, delay, redirect } = parseParams();
      log('params_parsed', { debug, hold, delay, redirect });

      // 3. Determinar redirect
      let redirectEnabled = config.redirectEnabled;
      if (config.allowOverride) {
        if (redirect === 'off') redirectEnabled = false;
        else if (redirect === 'on') redirectEnabled = true;
      }
      const shouldRedirect = redirectEnabled && !hold && !debug;

      // 4. Obter trc_id
      const trcId = getTrcId();
      if (!trcId || trcId === 'unknown') {
        log('error', { message: 'trc_id inválido' });
        return;
      }

      // 5. Coletar dados
      const queryParams = collectQueryParams(params);
      const origin = location.origin + location.pathname + location.search;

      // 6. Construir payload
      const payload = {
        event: 'pageview',
        trc_id: trcId,
        timestamp: new Date().toISOString(),
        source: {
          query: queryParams,
          user_agent: navigator.userAgent || '',
          origin
        }
      };
      log('payload_ready', { trc_id: trcId, length: JSON.stringify(payload).length });

      // 7. Enviar evento
      const send = () => {
        log('send_start', { trc_id: trcId });
        return window.__trc_net?.sendToTrcsub?.(payload)
          .then((ok) => { log('send_result', { ok }); return ok; })
          .catch((err) => { log('send_error', { error: String(err) }); return false; })
          || Promise.resolve(true);
      };

      // 8. Se não deve redirecionar, apenas enviar
      if (!shouldRedirect) {
        log('no_redirect', { reason: debug ? 'debug' : hold ? 'hold' : 'disabled' });
        send();
        return;
      }

      // 9. Construir link
      const link = buildLink(config.bot, config.mode, config.path, trcId);
      if (!link) {
        log('error', { message: 'Link inválido' });
        send();
        return;
      }
      log('link_built', { link });

      // 10. Delay configurado
      const minDelay = Math.max(0, delay || config.delayMin);
      const maxDelay = Math.max(minDelay, config.delayMax);

      // 11. Timeout de segurança
      const safetyTimeout = setTimeout(() => {
        log('redirect_timeout', { link });
        try { location.assign(link); } catch { location.href = link; }
      }, DEFAULTS.MAX_WAIT);

      // 12. Redirect com cleanup
      const doRedirect = () => {
        clearTimeout(safetyTimeout);
        log('redirect', { link });
        try { location.assign(link); } catch { location.href = link; }
      };

      // 13. Enviar e redirecionar
      if (window.__trc_net?.ensureDispatchThenRedirect) {
        window.__trc_net.ensureDispatchThenRedirect(send, doRedirect, minDelay, maxDelay);
      } else {
        send().finally(() => {
          setTimeout(doRedirect, minDelay);
        });
      }
    } catch (error) {
      log('fatal_error', { error: String(error) });
      // Tentar enviar evento de erro
      try {
        const trcId = getTrcId();
        if (trcId && trcId !== 'unknown') {
          window.__trc_net?.sendToTrcsub?.({
            event: 'pageview',
            trc_id: trcId,
            timestamp: new Date().toISOString(),
            error: true,
            error_message: String(error)
          });
        }
      } catch {}
    }
  }

  // Iniciar
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', main);
  } else {
    main();
  }
})();
